#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "rng.h"
#include "api.h"
#include "gmp.h"
#include "kaz_api.h"
#include "sha256.h"

void HashMsg(const unsigned char *msg, unsigned long long mlen, unsigned char buf[32])
{
    sha256_t hash;
	sha256_init(&hash);
	sha256_update(&hash, msg, mlen);
	sha256_update(&hash, msg, mlen);
	sha256_final(&hash, buf);
}

void KAZ_DS_OrderBase(mpz_t Modular,
                      mpz_t FiModular,
                      mpz_t Base,
                      mpz_t OrderBase)
{
    mpz_t tmp, G1;
    mpz_inits(tmp, G1, NULL);

    int nFiModular=KAZ_DS_GET_PFactors(FiModular);

    mpz_t *pFactors=malloc(nFiModular*sizeof(mpz_t));
    int *q=malloc(nFiModular*sizeof(int));
    int *e=malloc(nFiModular*sizeof(int));
    int *esimpan=malloc(nFiModular*sizeof(int));

    for(int i=0; i<nFiModular; i++) mpz_init(pFactors[i]);
    for(int i=0; i<nFiModular; i++) q[i]=0;
    for(int i=0; i<nFiModular; i++) e[i]=0;
    for(int i=0; i<nFiModular; i++) esimpan[i]=0;

    KAZ_DS_PFactors(FiModular, pFactors, q, e);

    for(int j=0; j<nFiModular; j++){
        for(int i=1; i<=e[j]; i++){
            mpz_set_ui(tmp, q[j]);
            mpz_pow_ui(tmp, tmp, i);
            mpz_divexact(tmp, FiModular, tmp);
            mpz_powm(tmp, Base, tmp, Modular);

            if(mpz_cmp_ui(tmp, 1)!=0){
                esimpan[j]=i-1;
                break;
            }
        }
    }

    mpz_set_ui(G1, 1);
    for(int j=0; j<nFiModular; j++){
        mpz_set_ui(tmp, q[j]);
        mpz_pow_ui(tmp, tmp, esimpan[j]);
        mpz_mul(G1, G1, tmp);
    }

    mpz_divexact(OrderBase, FiModular, G1);
}

void KAZ_DS_CRT(mpz_t ord,
                mpz_t *pfactors,
                mpz_t *candidatex,
                int nof,
                mpz_t crt)
{
    mpz_t tmp;
    mpz_init(tmp);

    mpz_t *b=malloc(nof*sizeof(mpz_t));
    mpz_t *binv=malloc(nof*sizeof(mpz_t));

    for(int i=0; i<nof; i++) mpz_init(b[i]);
    for(int i=0; i<nof; i++) mpz_init(binv[i]);
    for(int i=0; i<nof; i++) mpz_divexact(b[i], ord, pfactors[i]);
    for(int i=0; i<nof; i++){
        mpz_set(tmp, pfactors[i]);
        mpz_invert(binv[i], b[i], tmp);
    }

    mpz_set_ui(crt, 0);

    for(int i=0; i<nof; i++){
        mpz_set(tmp, candidatex[i]);
        mpz_mul(tmp, tmp, b[i]);
        mpz_mul(tmp, tmp, binv[i]);
        mpz_mod(tmp, tmp, ord);
        mpz_add(crt, crt, tmp);
        mpz_mod(crt, crt, ord);
    }
}

int KAZ_DS_GET_PFactors(mpz_t input)
{
    mpz_t inp, prod;
    mpz_inits(inp, prod, NULL);
    mpz_set(inp, input);
    mpz_set_ui(prod, 1);

    int div=2, count=0;
    int i=0;

    while(mpz_cmp_ui(inp, 1)>0){
        while(mpz_divisible_ui_p(inp, div)>0){
            count++;
            mpz_divexact_ui(inp, inp, div);
            mpz_mul_ui(prod, prod, div);
        }

        if(mpz_cmp_ui(prod, 1)>0){
            i++;
        }
        mpz_set_ui(prod, 1);
        div++;
        count=0;
    }

    return i;
}

void KAZ_DS_PFactors(mpz_t ord,
                     mpz_t *pfacs,
                     int *qlist,
                     int *elist)
{
    mpz_t inp, prod;
    mpz_inits(inp, prod, NULL);
    mpz_set(inp, ord);
    mpz_set_ui(prod, 1);

    int div=2, count=0;
    unsigned long long i=0;

    while(mpz_cmp_ui(inp, 1)>0){
        while(mpz_divisible_ui_p(inp, div)>0){
            count++;
            mpz_divexact_ui(inp, inp, div);
            mpz_mul_ui(prod, prod, div);
        }

        if(mpz_cmp_ui(prod, 1)>0){

            mpz_set(pfacs[i], prod);
            qlist[i]=div;
            elist[i]=count;
            i++;
        }
        mpz_set_ui(prod, 1);
        div++;
        count=0;
    }

    mpz_clear(inp);
}

char* KAZ_DS_MLOG(mpz_t n,
                 mpz_t ord,
                 mpz_t g,
                 mpz_t h,
                 mpz_t *pfactors,
                 int *qlist,
                 int *elist,
                 int saiz,
                 mpz_t kaz_crt)
{
    mpz_t tmp, tmp2, exp, G, H, j, delta, alpha, beta;
    mpz_inits(tmp, tmp2, exp, G, H, j, delta, alpha, beta, NULL);

    unsigned long kk=0;
    mpz_set_ui(kaz_crt, 0);

    int SAIZ=saiz;

    mpz_t *candidateX=malloc(SAIZ*sizeof(mpz_t));
    for(int i=0; i<SAIZ; i++) mpz_init(candidateX[i]);
    for(int i=0; i<SAIZ; i++) mpz_set_ui(candidateX[i], 0);
    for(int i=SAIZ-1; i>=0; i--){
        mpz_set_ui(delta, 1);
        kk=0;
        mpz_t *l=malloc(elist[i]*sizeof(mpz_t));

        for(int ii=0; ii<elist[i]; ii++) mpz_init(l[ii]);
        for(int j=1; j<=elist[i]; j++){
            mpz_divexact_ui(tmp, ord, qlist[i]);
            mpz_powm(alpha, g, tmp, n);

            if(elist[i]==1){
                mpz_powm(beta, h, tmp, n);
                mpz_t k, limits;
                mpz_inits(k, limits, NULL);
                mpz_set_ui(k, 0);
                mpz_set_ui(limits, qlist[i]);
                mpz_pow_ui(limits, limits, j);
                mpz_sub_ui(limits, limits, 1);

                while(mpz_cmp(k, limits)<1){
                    mpz_t tmp3; mpz_init(tmp3);
                    mpz_powm(tmp3, alpha, k, n);

                    if(mpz_cmp(tmp3, beta)==0){
                        break;
                    }
                    if(mpz_cmp(k, limits)>=0){
                        return "FAIL";
                    }
                    mpz_add_ui(k, k, 1);
                }

                mpz_set(candidateX[i], k);
            }else{
                mpz_t tmp5;
                mpz_init(tmp5);

                mpz_set_ui(tmp, qlist[i]);
                mpz_pow_ui(tmp, tmp, j); //q[i]^j
                mpz_divexact(tmp, ord, tmp); //n/q[i]^j
                mpz_powm(beta, h, tmp, n);
                mpz_neg(tmp, tmp);
                mpz_powm(tmp5, delta, tmp, n);
                mpz_mul(beta, beta, tmp5);
                mpz_mod(beta, beta, n);

                mpz_t k, limits;
                mpz_inits(k, limits, NULL);

                mpz_set_ui(k, 0);
                mpz_set_ui(limits, qlist[i]);
                mpz_pow_ui(limits, limits, j);
                mpz_sub_ui(limits, limits, 1);

                while(mpz_cmp(k, limits)<1){
                    mpz_t tmp3; mpz_init(tmp3);
                    mpz_powm(tmp3, alpha, k, n);

                    if(mpz_cmp(tmp3, beta)==0){
                        break;
                    }
                    if(mpz_cmp(k, limits)>=0){
                        return "FAIL";
                    }
                    mpz_add_ui(k, k, 1);
                }

                mpz_set(l[kk], k);

                mpz_t pwr, tmp4, tmp6;
                mpz_inits(pwr, tmp4, tmp6, NULL);

                mpz_set_ui(pwr, qlist[i]);
                mpz_pow_ui(pwr, pwr, (j-1));
                mpz_mul(pwr, l[kk], pwr);
                mpz_powm(tmp4, g, pwr, n);
                mpz_mul(delta, delta, tmp4);
                mpz_mod(delta, delta, n);
                mpz_add(candidateX[i], candidateX[i], pwr);
                kk++;
            }
        }

        mpz_mod(candidateX[i], candidateX[i], pfactors[i]);
    }

    int k = SAIZ;
    KAZ_DS_CRT(ord, pfactors, candidateX, k, kaz_crt);
}

void KAZ_DS_KeyGen(unsigned char *kaz_ds_verify_key,
                   unsigned char *kaz_ds_sign_key)
{
    mpz_t N, g, Gg, R, GRg, q, ALPHA, GRgq, V1, q2, tmp, V2;
    mpz_inits(N, g, Gg, R, GRg, q, ALPHA, GRgq, V1, q2, tmp, V2, NULL);

    //1) Get all system parameters
    mpz_set_str(N, KAZ_DS_SP_N, 10);
    mpz_set_str(g, KAZ_DS_SP_G, 10);
    mpz_set_str(Gg, KAZ_DS_SP_Gg, 10);
    mpz_set_str(R, KAZ_DS_SP_R, 10);
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(q, KAZ_DS_SP_Q, 10);

    int ss=KAZ_DS_SP_K;
    int nphiGg=KAZ_DS_SP_nPHIGg;

    //2) Generate ALPHA & (V1, V2)
    KAZ_DS_RANDOM(nphiGg-2, nphiGg-1, ALPHA);

    mpz_mul(GRgq, GRg, q);
    mpz_mod(V1, ALPHA, GRgq);

    mpz_mul(q2, q, q);
    mpz_powm_ui(tmp, ALPHA, 4, q2);

    //mpz_set_str(ALPHA, "6219349725576856466711287278115657230584282733446116779818524189672927710798454635906989293319781773998651702685971", 10);
    //mpz_set_str(V1, "9554529848552414597631958498266724306417528642082148025735730057971", 10);
    //mpz_set_str(tmp, "38809782345366891487567799333792723080740688466013621996732408311706293940433", 10);

    size_t TMPSIZE=mpz_sizeinbase(tmp, 16);

	unsigned char *TMPBYTE=(unsigned char*) malloc(TMPSIZE*sizeof(unsigned char));
	mpz_export(TMPBYTE, &TMPSIZE, 1, sizeof(char), 0, 0, tmp);

    unsigned char buf[32]={0};
    HashMsg(TMPBYTE, TMPSIZE, buf);

    mpz_import(V2, 32, 1, sizeof(char), 0, 0, buf);

    //3) Set kaz_ds_sign_key=(ALPHA, V2) & kaz_ds_verify_key=(V1, V2, V3)
    size_t ALPHASIZE=mpz_sizeinbase(ALPHA, 16);
	size_t V1SIZE=mpz_sizeinbase(V1, 16);
	size_t V2SIZE=mpz_sizeinbase(V2, 16);

	unsigned char *ALPHABYTE=(unsigned char*) malloc(ALPHASIZE*sizeof(unsigned char));
	mpz_export(ALPHABYTE, &ALPHASIZE, 1, sizeof(char), 0, 0, ALPHA);

	unsigned char *V1BYTE=(unsigned char*) malloc(V1SIZE*sizeof(unsigned char));
	mpz_export(V1BYTE, &V1SIZE, 1, sizeof(char), 0, 0, V1);

	unsigned char *V2BYTE=(unsigned char*) malloc(V2SIZE*sizeof(unsigned char));
	mpz_export(V2BYTE, &V2SIZE, 1, sizeof(char), 0, 0, V2);

	for(int i=0; i<CRYPTO_SECRETKEYBYTES; i++) kaz_ds_sign_key[i]=0;

	int je=CRYPTO_SECRETKEYBYTES-1;
	for(int i=ALPHASIZE-1; i>=0; i--){
		kaz_ds_sign_key[je]=ALPHABYTE[i];
		je--;
	}

	for(int i=0; i<CRYPTO_PUBLICKEYBYTES; i++) kaz_ds_verify_key[i]=0;

	je=CRYPTO_PUBLICKEYBYTES-1;
	for(int i=V2SIZE-1; i>=0; i--){
		kaz_ds_verify_key[je]=V2BYTE[i];
		je--;
	}

	je=CRYPTO_PUBLICKEYBYTES-KAZ_DS_V2BYTES-1;
	for(int i=V1SIZE-1; i>=0; i--){
		kaz_ds_verify_key[je]=V1BYTE[i];
		je--;
	}

    //gmp_printf("ALPHA=%Zd\nV1=%Zd\nV2=%Zd\n", ALPHA, V1,V2);
    //gmp_printf("ALPHA=%Zd\nV1=%Zd\nV2=%Zd\nV3=%Zd\n", ALPHA, V1, V2, V3);
	mpz_clears(N, g, Gg, R, GRg, q, ALPHA, GRgq, V1, q2, tmp, V2, NULL);
}

int KAZ_DS_SIGNATURE(unsigned char *signature,
                     unsigned long long *signlen,
                     const unsigned char *m,
                     unsigned long long mlen,
                     const unsigned char *sk)
{
    mpz_t N, g, Gg, R, GRg, q, ALPHA, hashValue, q2, GRgq2, r, rinverse, S1, GS1, GS12, tmp, S2;
    mpz_inits(N, g, Gg, R, GRg, q, ALPHA, hashValue, q2, GRgq2, r, rinverse, S1, GS1, GS12, tmp, S2, NULL);

    //1) Get all system parameters
    mpz_set_str(N, KAZ_DS_SP_N, 10);
    mpz_set_str(g, KAZ_DS_SP_G, 10);
    mpz_set_str(Gg, KAZ_DS_SP_Gg, 10);
    mpz_set_str(R, KAZ_DS_SP_R, 10);
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(q, KAZ_DS_SP_Q, 10);

    int ss=KAZ_DS_SP_K;
    int nphiGg=KAZ_DS_SP_nPHIGg;

    //2) Get kaz_ds_sign_key=ALPHA
    int ALPHASIZE=0;
	for(int i=0; i<KAZ_DS_ALPHABYTES; i++){
        if((int)sk[i]==0) ALPHASIZE++;
        else break;
	}

	unsigned char *ALPHABYTE=(unsigned char*) malloc((KAZ_DS_ALPHABYTES-ALPHASIZE)*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_ALPHABYTES-ALPHASIZE; i++) ALPHABYTE[i]=0;

	for(int i=0; i<KAZ_DS_ALPHABYTES-ALPHASIZE; i++){ALPHABYTE[i]=sk[i+ALPHASIZE];}

    mpz_import(ALPHA, KAZ_DS_ALPHABYTES-ALPHASIZE, 1, sizeof(char), 0, 0, ALPHABYTE);

	//gmp_printf("ALPHA=%Zd\n", ALPHA);

	//3) Generate the hash value of the message
	unsigned char buf[CRYPTO_BYTES]={0};
    HashMsg(m, mlen, buf);

    mpz_import(hashValue, CRYPTO_BYTES, 1, sizeof(char), 0, 0, buf);
    //mpz_set_str(hashValue, "106540710009122782407045432239366525580800704361160875832842826683924836413598", 10);

    mpz_mul(q2, q, q);
    mpz_mul(GRgq2, GRg, q2);

    //4) Generate S1, S2
    do{
        do{
            KAZ_DS_RANDOM(nphiGg-2, nphiGg-1, r);

            mpz_mod(S1, r, GRgq2);
            mpz_gcd(GS1, S1, GRg);
            mpz_gcd(GS12, r, GRg);
        }while(mpz_cmp_ui(GS12, 100)<0);

        mpz_div(tmp, r, GS1);
        mpz_invert(rinverse, tmp, GRgq2);
        mpz_powm(tmp, ALPHA, S1, GRgq2);
        mpz_add(tmp, tmp, hashValue);
        mpz_mod(tmp, tmp, GRgq2);
        //mpz_mul(tmp, tmp, GS1);
        mpz_mod(tmp, tmp, GRgq2);
        mpz_mul(tmp, tmp, rinverse);
        mpz_mod(S2, tmp, GRgq2);
    }while(mpz_cmp_ui(S2, 0)==0);

    //gmp_printf("S1=%Zd\nS2=%Zd\nh=%Zd\n", S1, S2, hashValue);
    //mpz_set_str(S1, "2016685929212656579026119527399880093855006856668979238281265153686599579364974597757097832207108705144200", 10);
    //mpz_set_str(S2, "272706862338737495266999022230365081789357689271907285550915611608292353350616368156736549702739210404079", 10);
    //5) Set signature=(S1, S2)
    size_t S1SIZE=mpz_sizeinbase(S1, 16);
	size_t S2SIZE=mpz_sizeinbase(S2, 16);

    for(int i=0; i<mlen+KAZ_DS_S1BYTES+KAZ_DS_S2BYTES; i++) signature[i]=0;

	unsigned char *S1BYTE=(unsigned char*) malloc(S1SIZE*sizeof(unsigned char));
	mpz_export(S1BYTE, &S1SIZE, 1, sizeof(char), 0, 0, S1);

	unsigned char *S2BYTE=(unsigned char*) malloc(S2SIZE*sizeof(unsigned char));
	mpz_export(S2BYTE, &S2SIZE, 1, sizeof(char), 0, 0, S2);

	int je=mlen+KAZ_DS_S2BYTES+KAZ_DS_S1BYTES-1;
	for(int i=mlen-1; i>=0; i--){
		signature[je]=m[i];
		je--;
	}

	je=KAZ_DS_S2BYTES+KAZ_DS_S1BYTES-1;
	for(int i=S2SIZE-1; i>=0; i--){
		signature[je]=S2BYTE[i];
		je--;
	}

    je=KAZ_DS_S1BYTES-1;
	for(int i=S1SIZE-1; i>=0; i--){
		signature[je]=S1BYTE[i];
		je--;
	}

	*signlen=mlen+KAZ_DS_S1BYTES+KAZ_DS_S2BYTES;

    free(S1BYTE);
    free(S2BYTE);

	mpz_clears(N, g, Gg, R, GRg, q, ALPHA, hashValue, q2, GRgq2, r, rinverse, S1, GS1, GS12, tmp, S2, NULL);

    return 0;
}

int KAZ_DS_VERIFICATION(unsigned char *m,
                        unsigned long long *mlen,
                        const unsigned char *sm,
                        unsigned long long smlen,
                        const unsigned char *pk)
{
    mpz_t N, g, Gg, R, GRg, q, GS1r, aF, q2, GRgq2, tmp, S1inverse, hashValue, V1, V2, S1, S2, w0, w1, w2, w3, w4, w5, w6, w7, w8, w9, y1, y2, z0, z1, phiq2o2;
    mpz_inits(N, g, Gg, R, GRg, q, GS1r, aF, q2, GRgq2, tmp, S1inverse, hashValue, V1, V2, S1, S2, w0, w1, w2, w3, w4, w5, w6, w7, w8, w9, y1, y2, z0, z1, phiq2o2, NULL);

    //1) Get all system parameters
    mpz_set_str(N, KAZ_DS_SP_N, 10);
    mpz_set_str(g, KAZ_DS_SP_G, 10);
    mpz_set_str(Gg, KAZ_DS_SP_Gg, 10);
    mpz_set_str(R, KAZ_DS_SP_R, 10);
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(q, KAZ_DS_SP_Q, 10);
    mpz_set_str(phiq2o2, KAZ_DS_SP_PHIQ2O2, 10);

    int ss=KAZ_DS_SP_K;
    int nphiGg=KAZ_DS_SP_nPHIGg;

    //2) Get kaz_ds_verify_key=(V1, V2)
	int V1SIZE=0;
	for(int i=0; i<KAZ_DS_V1BYTES; i++){
        if((int)pk[i]==0) V1SIZE++;
        else break;
	}

	int V2SIZE=0;
	for(int i=KAZ_DS_V1BYTES; i<CRYPTO_PUBLICKEYBYTES; i++){
        if((int)pk[i]==0) V2SIZE++;
        else break;
	}

	unsigned char *V1BYTE=(unsigned char*) malloc((KAZ_DS_V1BYTES-V1SIZE)*sizeof(unsigned char));
	unsigned char *V2BYTE=(unsigned char*) malloc((KAZ_DS_V2BYTES-V2SIZE)*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_V1BYTES-V1SIZE; i++) V1BYTE[i]=0;
	for(int i=0; i<KAZ_DS_V2BYTES-V2SIZE; i++) V2BYTE[i]=0;

	for(int i=0; i<KAZ_DS_V1BYTES-V1SIZE; i++){V1BYTE[i]=pk[i+V1SIZE];}
	for(int i=0; i<KAZ_DS_V2BYTES-V2SIZE; i++){V2BYTE[i]=pk[i+KAZ_DS_V1BYTES+V2SIZE];}

	mpz_import(V1, KAZ_DS_V1BYTES-V1SIZE, 1, sizeof(char), 0, 0, V1BYTE);
	mpz_import(V2, KAZ_DS_V2BYTES-V2SIZE, 1, sizeof(char), 0, 0, V2BYTE);

	//gmp_printf("V1=%Zd\nV2=%Zd\n", V1, V2);

    //3) Get signature=(S1, S2, m)
    int S1SIZE=0;
	for(int i=0; i<KAZ_DS_S1BYTES; i++){
        if((int)sm[i]==0) S1SIZE++;
        else break;
	}

	int S2SIZE=0;
	for(int i=KAZ_DS_S1BYTES; i<KAZ_DS_S1BYTES+KAZ_DS_S2BYTES; i++){
        if((int)sm[i]==0) S2SIZE++;
        else break;
	}

	int MSIZE=0;
	for(int i=KAZ_DS_S1BYTES+KAZ_DS_S2BYTES; i<smlen; i++){
        if((int)sm[i]==0) MSIZE++;
        else break;
	}

	unsigned char *S1BYTE=(unsigned char*) malloc((KAZ_DS_S1BYTES-S1SIZE)*sizeof(unsigned char));
	unsigned char *S2BYTE=(unsigned char*) malloc((KAZ_DS_S2BYTES-S2SIZE)*sizeof(unsigned char));
	unsigned char *MBYTE=(unsigned char*) malloc((smlen-(KAZ_DS_S1BYTES+KAZ_DS_S2BYTES)-MSIZE)*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_S1BYTES-S1SIZE; i++) S1BYTE[i]=0;
	for(int i=0; i<KAZ_DS_S2BYTES-S2SIZE; i++) S2BYTE[i]=0;
	for(int i=0; i<smlen-(KAZ_DS_S1BYTES+KAZ_DS_S2BYTES)-MSIZE; i++) MBYTE[i]=0;

	for(int i=0; i<KAZ_DS_S1BYTES-S1SIZE; i++){S1BYTE[i]=sm[i+S1SIZE];}
	for(int i=0; i<KAZ_DS_S2BYTES-S2SIZE; i++){S2BYTE[i]=sm[i+(KAZ_DS_S1BYTES+S2SIZE)];}
	for(int i=0; i<smlen-(KAZ_DS_S1BYTES+KAZ_DS_S2BYTES)-MSIZE; i++){MBYTE[i]=sm[i+(KAZ_DS_S1BYTES+KAZ_DS_S2BYTES+MSIZE)];}

    mpz_import(S1, KAZ_DS_S1BYTES-S1SIZE, 1, sizeof(char), 0, 0, S1BYTE);
	mpz_import(S2, KAZ_DS_S2BYTES-S2SIZE, 1, sizeof(char), 0, 0, S2BYTE);

	//4) Compute the hash value of the message
    unsigned char buf[CRYPTO_BYTES]={0}; int len=smlen-KAZ_DS_S1BYTES-KAZ_DS_S2BYTES-MSIZE;
    HashMsg(MBYTE, len, buf);

    mpz_import(hashValue, CRYPTO_BYTES, 1, sizeof(char), 0, 0, buf);
    //mpz_set_str(hashValue, "106540710009122782407045432239366525580800704361160875832842826683924836413598", 10);

    //gmp_printf("S1=%Zd\nS2=%Zd\nh=%Zd\nSelesai\n\n", S1, S2, hashValue);

    //5) Filter 1
    mpz_gcd(GS1r, S1, GRg);
    mpz_mod(aF, V1, GRg);

    mpz_mul(q2, q, q);
    mpz_mul(GRgq2, GRg, q2);

    mpz_invert(S1inverse, S1, GRgq2);
    mpz_powm(tmp, V1, S1, GRgq2);
    mpz_add(tmp, tmp, hashValue);
    mpz_mod(tmp, tmp, GRgq2);
    mpz_mul(tmp, tmp, GS1r);
    mpz_mod(tmp, tmp, GRgq2);
    mpz_mul(tmp, tmp, S1inverse);
    mpz_mod(w0, tmp, GRgq2);

    mpz_sub(w1, S2, w0);

    if(mpz_cmp_ui(w1, 0)==0){//printf("w1==0\n");
        return -4; //REJECT SIGNATURE
    }

    //6) Filter 2
    mpz_powm(tmp, aF, S1, GRgq2);
    mpz_add(tmp, tmp, hashValue);
    mpz_mod(tmp, tmp, GRgq2);
    mpz_mul(tmp, tmp, GS1r);
    mpz_mod(tmp, tmp, GRgq2);
    mpz_mul(tmp, tmp, S1inverse);
    mpz_mod(w2, tmp, GRgq2);

    mpz_sub(w3, S2, w2);

    if(mpz_cmp_ui(w3, 0)==0){//printf("w3==0\n");
        return -4; //REJECT SIGNATURE
    }

    //7) Filter 3
    mpz_mul(tmp, S1, S2);
    mpz_mod(w4, tmp, q);
    mpz_mul(tmp, GS1r, hashValue);
    mpz_mod(tmp, tmp, q);
    mpz_sub(w4, w4, tmp);
    mpz_mod(w4, w4, q);

    mpz_powm(w5, V1, S1, q);
    mpz_mul(w5, w5, GS1r);
    mpz_mod(w5, w5, q);

    mpz_sub(w6, w4, w5); //gmp_printf("w4=%Zd\nw5=%Zd\n", w4, w5);

    if(mpz_cmp_ui(w6, 0)!=0){//printf("w6!=0\n");
        return -4; //REJECT SIGNATURE
    }

    //8) Filter 4
    mpz_invert(w7, S1, phiq2o2);
    mpz_mul_ui(w7, w7, 2);
    mpz_mod(w7, w7, phiq2o2);

    mpz_mul(w8, S1, S2);
    mpz_mod(w8, w8, q2);
    mpz_mul(tmp, GS1r, hashValue);
    mpz_mod(tmp, tmp, q2);
    mpz_sub(w8, w8, tmp);
    mpz_mod(w8, w8, q2);
    mpz_invert(tmp, GS1r, q2);
    mpz_mul(w8, w8, tmp);
    mpz_mod(w8, w8, q2);
    mpz_mul_ui(tmp, w7, 2);
    mpz_powm(w8, w8, tmp, q2);

    size_t W8SIZE=mpz_sizeinbase(w8, 16);

	unsigned char *W8BYTE=(unsigned char*) malloc(W8SIZE*sizeof(unsigned char));
	mpz_export(W8BYTE, &W8SIZE, 1, sizeof(char), 0, 0, w8);

    unsigned char bufw8[32]={0};
    HashMsg(W8BYTE, W8SIZE, bufw8);

    mpz_import(w8, 32, 1, sizeof(char), 0, 0, bufw8);

    mpz_sub(w9, w8, V2);

    if(mpz_cmp_ui(w9, 0)!=0){//printf("w9!=0\n");
        return -4; //REJECT SIGNATURE
    }

    //9) Verify signature
    mpz_mul(tmp, S1, S2);
    mpz_powm(z0, R, tmp, Gg);
	mpz_powm(y1, g, z0, N);

	mpz_powm(tmp, V1, S1, GRg);
	mpz_add(tmp, tmp, hashValue);
	mpz_mul(tmp, tmp, GS1r);
	mpz_mod(tmp, tmp, GRg);
	mpz_powm(z1, R, tmp, Gg);
	mpz_powm(y2, g, z1, N);

	if(mpz_cmp(y1, y2)==0){
        memcpy(m, MBYTE, len);
        *mlen=len;

        free(S1BYTE);
        free(S2BYTE);
        free(MBYTE);

        mpz_clears(N, g, Gg, R, GRg, q, GS1r, aF, q2, GRgq2, tmp, S1inverse, hashValue, w0, w1, w2, w3, w4, w5, w6, w7, w8, w9, y1, y2, z0, z1, phiq2o2, NULL);

        return 0;
	}

	/*gmp_printf("V1=%Zd\nV2=%Zd\nV3=%Zd\n", V1, V2, V3);
	gmp_printf("S1=%Zd\nS2=%Zd\nS3=%Zd\nh=%Zd\n", S1, S2, S3, hashValue);
	gmp_printf("w0=%Zd\nw1=%Zd\n", w0, w1);
	gmp_printf("y1=%Zd\ny2=%Zd\n\n", y1, y2);*/

	return -4;
}

void KAZ_DS_RANDOM(int lb,
                   int ub,
                   mpz_t out){
    ////time_t seed;
	//srand((unsigned) time(&seed));

	//mpz_t randNUM; mpz_init(randNUM);

	mpz_t lbound, ubound;

	gmp_randstate_t gmpRandState;
    gmp_randinit_mt(gmpRandState);
	mpz_inits(lbound, ubound, NULL);

	mpz_ui_pow_ui(lbound, 2, lb);
	mpz_ui_pow_ui(ubound, 2, ub);

	unsigned int sd=100000;

	do{
		// initialize state for a Mersenne Twister algorithm. This algorithm is fast and has good randomness properties.

		//gmp_randseed_ui(gmpRandState, sd);
		gmp_randseed_ui(gmpRandState, rand()+sd);
		mpz_urandomb(out, gmpRandState, ub);
		sd+=1;
	}while((mpz_cmp(out, lbound) == -1) || (mpz_cmp(out, ubound) == 1));

	// empty the memory location for the random generator state
    //gmp_randclear(gmpRandState);
}

/*void KAZ_DS_RANDOM(int lb,
                   int ub,
                   mpz_t out){
    time_t seed;
	srand((unsigned) time(&seed));

	mpz_t randNUM; mpz_init(randNUM);

	mpz_t lbound, ubound;

	gmp_randstate_t gmpRandState;

	mpz_inits(lbound, ubound, NULL);

	mpz_ui_pow_ui(lbound, 2, lb);
	mpz_ui_pow_ui(ubound, 2, ub);

	do{
		// initialize state for a Mersenne Twister algorithm. This algorithm is fast and has good randomness properties.
		gmp_randinit_mt(gmpRandState);
		gmp_randseed_ui(gmpRandState, rand());
		mpz_urandomb(out, gmpRandState, ub);
	}while((mpz_cmp(out, lbound) == -1) || (mpz_cmp(out, ubound) == 1));

	// empty the memory location for the random generator state
    gmp_randclear(gmpRandState);
}*/
